#! /usr/bin/env python3
# -*- coding: utf-8 -*-

#-------------------------------------------------------------------------------------------
# Issue: https://www.klayout.de/forum/discussion/2834/
#
#     Sub: Feasibility and Approach for Opening GDSII Files in KLayout via Plugin
#     Posted by: Simran (simransingh14)
#
# Author of this PYA: Kazzz-S
#      Last modified: 2026-01-12
#-------------------------------------------------------------------------------------------
import os
import sys
import platform
import math
import types
import json
import time
import tempfile
import shutil
import traceback
import pya
from   pathlib import Path
from   typing import Optional, Any, Dict, Tuple
import xml.etree.ElementTree as ET
from   urllib.parse import urljoin, urlparse, urlunparse
from   urllib.request import Request, urlopen
from   urllib.error import HTTPError, URLError

MyDir = os.path.dirname(os.path.abspath(__file__))
sys.path.append(MyDir)

#=======================================================================================
# Set the system global variables  ===> see "Launcher.py" by ChatGPT
#=======================================================================================
APP_NAME = "GDS Launcher2"
ORG_NAME = "Forum2834"
APP_SETTINGS_NAME = "GDSLauncher2"

DEFAULT_KLAYOUT_CMD = "klayout"

BASE_DIR = Path(__file__).resolve().parent
TESTDATA_DIR = BASE_DIR / "testdata"

LOCAL_MAP = {
  "TEST-GDS-001": {
    "gds": TESTDATA_DIR / "sample.gds",
    "lyp": TESTDATA_DIR / "sample.lyp",
  },
  "TEST-GDS-002": {
    "gds": TESTDATA_DIR / "cobra3b.gds",
    "lyp": TESTDATA_DIR / "cobra3b.lyp",
  },
}

API_LOGIN = "/api/auth/login"
API_LOOKUP = "/api/layouts/by-ref/{gdsRefId}"

TEMP_PREFIX = "gdslauncher_"
DEFAULT_CLEANUP_DAYS = 7

#=======================================================================================
# Core function definitions
#=======================================================================================
def CheckKLayoutVersion():
  """
  Check and set the KLayout version.

  Parameters
  ----------
  None

  Returns
  -------
  None
  """
  global KLayoutVernum

  verstring    = pya.Application().instance().version() # like "KLayout 0.25.9"
  name, vernum = verstring.split(" ")
  vMajor       = "0"
  vMinor       = "0"
  vRev         = "0"
  try:
    vMajor, vMinor, vRev = vernum.split(".")
  except:
    try:
      vMajor, vMinor = vernum.split(".")  # like "KLayout 0.25"
    except:
      pass
  version = 1000*int(vMajor) + 100*int(vMinor) + 1*int(vRev) # >= 3005
  if not version >= 3005:
    expmsg = '! This plugin assumes KLayout >= 0.30.5 but <%s> is used' % vernum
    raise Exception(expmsg)
  else:
    KLayoutVernum = vernum

def InitGlobals():
  """
  Initialize the global objects.

  Parameters
  ----------
  None

  Returns
  -------
  None
  """
  global MainWindow       # the main window
  global MainMenu         # the main menu
  global Action           # the action
  global ToolGUI          # the tool's GUI (*.ui)
  global ToolResource     # the tool's resource (*.rcc)
  global Dialog           # the main dialog
  global Settings         # the settings
  global DebugWithMock    # the debug flag

  VerboseLevel  = 0    # default=0; set a greater number for debugging
  MainWindow    = None
  MainMenu      = None
  Action        = None
  ToolGUI       = "Forum2834.ui"
  ToolResource  = "Forum2834.rcc" # not provided because the UI design is not fancy
  Dialog        = None
  Settings      = None
  DebugWithMock = False

def KLayout_REST_API():
  """
  Construct and show the main GUI for KLayout-REST_API demonstration.

  Parameters
  ----------
  None

  Returns
  -------
  None
  """
  global Dialog
  global Settings

  #---------------------------------------------------------------------------------------
  # [1] Import *.ui file
  #
  #     Regarding the handling of a resource file, refer to:
  #       1) https://www.klayout.de/forum/discussion/comment/2896#Comment_2896
  #       2) https://github.com/KLayout/klayout/issues/730
  #---------------------------------------------------------------------------------------
  #pya.QResource().registerResource_file( MyDir + "/" + ToolResource )
  ui_file = pya.QFile.new( MyDir + "/" + ToolGUI )
  ui_file.open( pya.QIODevice.ReadOnly )
  #ui_file.open( pya.QIODevice_OpenModeFlag.ReadOnly )
  #Dialog = pya.QFormBuilder().load(ui_file, pya.Application.instance().main_window())
  loader = pya.QUiLoader()
  Dialog = loader.load( ui_file, pya.Application.instance().main_window() )
  ui_file.close()

  #---------------------------------------------------------------------------------------
  # [2] Set SLOTs for
  #       "pushButton_RecipeXML.clicked()"
  #       "checkBox_Debug.clicked()"
  #       "pushButton_CopyURL.clicked()"
  #       "pushButton_Clear.clicked()"
  #       "pushButton_Cancel.clicked()"
  #       "pushButton_Open.clicked()"
  #
  #       "lineEdit_AppURL.returnPressed()"
  #       "lineEdit_UserName.returnPressed()"
  #       "lineEdit_Password.returnPressed()"
  #---------------------------------------------------------------------------------------
  # Disable the default actions of push buttons
  Dialog.pushButton_RecipeXML.default     = False
  Dialog.pushButton_RecipeXML.autoDefault = False
  Dialog.pushButton_CopyURL  .default     = False
  Dialog.pushButton_CopyURL  .autoDefault = False
  Dialog.pushButton_Clear    .default     = False
  Dialog.pushButton_Clear    .autoDefault = False
  Dialog.pushButton_Cancel   .default     = False
  Dialog.pushButton_Cancel   .autoDefault = False
  Dialog.pushButton_Open     .default     = False
  Dialog.pushButton_Open     .autoDefault = False

  Dialog.pushButton_RecipeXML.clicked     ( types.MethodType( LoadRecipeXML,     Dialog ) )
  Dialog.checkBox_Debug.clicked           ( types.MethodType( DebugWithMockGDS,  Dialog ) )
  Dialog.pushButton_CopyURL.clicked       ( types.MethodType( CopyURL2Clipboard, Dialog ) )
  Dialog.pushButton_Clear.clicked         ( types.MethodType( ClearDialog,       Dialog ) )
  Dialog.pushButton_Cancel.clicked        ( types.MethodType( CancelDialog,      Dialog ) )
  Dialog.pushButton_Open.clicked          ( types.MethodType( OpenInKLayout,     Dialog ) )

  Dialog.lineEdit_AppURL  .returnPressed(lambda: LineEditEntered(Dialog, "AppURL"))
  Dialog.lineEdit_UserName.returnPressed(lambda: LineEditEntered(Dialog, "UserName"))
  Dialog.lineEdit_Password.returnPressed(lambda: LineEditEntered(Dialog, "Password"))

  #---------------------------------------------------------------------------------------
  # [3] Make lineEdit_RefID read-only depending on OS; already set in .ui
  #---------------------------------------------------------------------------------------
  """
  current_os = platform.system()
  if current_os in ["Linux", "Darwin"]:
    Dialog.lineEdit_RefID.readOnly = True
    Dialog.lineEdit_RefID.toolTip  = "Read-only"
  """

  #---------------------------------------------------------------------------------------
  # [4] Clear the message label and restore the settings
  #---------------------------------------------------------------------------------------
  Dialog.checkBox_Debug.checkState = pya.Qt_CheckState.Unchecked
  DebugWithMock = False
  Dialog.label_Message.text = ""
  restore_settings(Dialog)

  #---------------------------------------------------------------------------------------
  # [5] Show (execute) the dialog as "modeless", which is the default.
  #        exec_() is PYA's special method not to confuse with Python's exec(), which is
  #        always "modal"
  #---------------------------------------------------------------------------------------
  # Dialog.exec_()
  Dialog.show()

def RegisterAction():
  """
  Register this tool to the main menu.

  Parameters
  ----------
  None

  Returns
  -------
  None
  """
  global MainWindow
  global MainMenu
  global Action

  MainWindow   = pya.MainWindow.instance()
  MainMenu     = MainWindow.menu()
  Action       = pya.Action()
  Action.title = "KLayout with REST API"
  Action.on_triggered( KLayout_REST_API )
  MainMenu.insert_item( "forum2834.end", "KLayout_REST_API", Action )

#=======================================================================================
# BEGIN SLOT definitions
#=======================================================================================
def LoadRecipeXML(dialog):
  """
  [Slot] for the "pushButton_RecipeXML.clicked()" SIGNAL.

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance that receives the signal.

  Returns
  -------
  None
  """

  # [1] Show a file selection dialog and get the path to a single XML file
  file_dialog = pya.QFileDialog(pya.MainWindow.instance())
  file_dialog.setWindowTitle("Select a Recipe XML")
  file_dialog.setNameFilter("Recipe XML (*.xml)")
  file_dialog.setFileMode(pya.QFileDialog.ExistingFile)

  if file_dialog.exec_():
    selected_files = file_dialog.selectedFiles()
    if selected_files:
      xml_path = selected_files[0]

      dialog.lineEdit_RecipeXML.text = xml_path
      if xml_path:
        gdsrefid = parse_xml_for_gds_id(xml_path)
        dialog.lineEdit_RefID.text = gdsrefid
  else:
    pya.Logger.info("Recipe XML selection was cancelled.")
    print("Cancelled")

def DebugWithMockGDS( dialog, status ):
  """
  [Slot] for the "checkBox_Debug.clicked()" SIGNAL.

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance that receives the signal.
  status : boolean
    True if the widget is checked; False, otherwise

  Returns
  -------
  None
  """
  if status == False:
    dialog.label_AppURL      .setEnabled(True)
    dialog.lineEdit_AppURL   .setEnabled(True)
    dialog.pushButton_CopyURL.setEnabled(True)
    dialog.label_UserName    .setEnabled(True)
    dialog.lineEdit_UserName .setEnabled(True)
    dialog.label_Password    .setEnabled(True)
    dialog.lineEdit_Password .setEnabled(True)
    dialog.label_Message     .text    = ""
    dialog.label_Message     .enabled = True
    dialog.pushButton_Open   .text    = "Login && Open in KLayout"
  else:
    dialog.label_AppURL      .setEnabled(False)
    dialog.lineEdit_AppURL   .setEnabled(False)
    dialog.pushButton_CopyURL.setEnabled(False)
    dialog.label_UserName    .setEnabled(False)
    dialog.lineEdit_UserName .setEnabled(False)
    dialog.label_Password    .setEnabled(False)
    dialog.lineEdit_Password .setEnabled(False)
    dialog.label_Message     .text    = "Mock mode: no login required."
    dialog.label_Message     .enabled = True
    dialog.pushButton_Open   .text    = "Open in KLayout"

def CopyURL2Clipboard(dialog):
  """
  [Slot] for the "pushButton_CopyURL.clicked()" SIGNAL.

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance that receives the signal.

  Returns
  -------
  None
  """
  # The getter is NOT dialog.lineEdit_AppURL.text() but .text in PYA.
  text = (dialog.lineEdit_AppURL.text or "").strip()
  if not text:
    pya.MessageBox.warning(
      "Copy URL",
      "URL is empty",
      pya.MessageBox.Ok
    )
    return

  cb = pya.QApplication.clipboard()
  cb.setText(text, pya.QClipboard_Mode.Clipboard)

def LineEditEntered(dialog, widget_name):
  """
  [Slot] for the SIGNALs
          "lineEdit_AppURL  .returnPressed()"
          "lineEdit_UserName.returnPressed()"
          "lineEdit_Password.returnPressed()"

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance that receives the signal.
  widget_name: str
    A string to identify the lineEdit_xxx that call this method.

  Returns
  -------
  None
  """
  if widget_name == "AppURL":
    dialog.lineEdit_UserName.setFocus()
    dialog.lineEdit_UserName.selectAll()
  elif widget_name == "UserName":
    dialog.lineEdit_Password.setFocus()
    dialog.lineEdit_Password.selectAll()
  elif widget_name == "Password":
    dialog.pushButton_Open.setFocus()
  else:
    dialog.pushButton_Open.setFocus()

def ClearDialog(dialog):
  """
  [Slot] for the "pushButton_Clear.clicked()" SIGNAL.

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance that receives the signal.

  Returns
  -------
  None
  """
  dialog.lineEdit_RecipeXML.text       = ""
  dialog.lineEdit_RefID    .text       = ""
  dialog.checkBox_Debug    .checkState = pya.Qt_CheckState.Unchecked
  dialog.lineEdit_AppURL   .text       = ""
  dialog.lineEdit_UserName .text       = ""
  dialog.lineEdit_Password .text       = ""
  dialog.label_Message     .text       = ""

def CancelDialog(dialog):
  """
  [Slot] for the "pushButton_Cancel.clicked()" SIGNAL.

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance that receives the signal.

  Returns
  -------
  None
  """
  save_settings(dialog)
  dialog.reject()

def OpenInKLayout(dialog):
  """
  [Slot] for the "pushButton_Open.clicked()" SIGNAL.

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance that receives the signal.

  Returns
  -------
  None
  """
  debug = dialog.checkBox_Debug.checkState

  # [1] Debug mode
  if debug == pya.Qt_CheckState.Checked:
    refID = dialog.lineEdit_RefID.text
    try:
      gds_lyp = LOCAL_MAP[refID]
    except KeyError:
      pya.MessageBox.warning(
        "Referent ID",
        f"Unknown {refID}",
        pya.MessageBox.Ok
      )
      return
    else:
      load_design_files(gds=gds_lyp["gds"], lyp=gds_lyp["lyp"])
      return

  # [2] Normal mode
  try:
    base_url = (dialog.lineEdit_AppURL.text or "").strip()
    username = (dialog.lineEdit_UserName.text or "").strip()
    password = dialog.lineEdit_Password.text or ""
    refID    = (dialog.lineEdit_RefID.text or "").strip()

    if not base_url:
      raise RuntimeError("Base URL is empty.")
    if not username:
      raise RuntimeError("User name is empty.")
    if not password:
      raise RuntimeError("Password is empty.")
    if not refID:
      raise RuntimeError("Ref ID is empty.")

    dialog.label_Message.text    = "REAL: logging in ..."
    dialog.label_Message.enabled = True

    client = AuthClient(base_url, timeout=30)
    client.login(username, password)

    dialog.label_Message.text = "REAL: looking up GDS/LYP ..."
    lookup_path = API_LOOKUP.format(gdsRefId=refID)
    lookup_json = client.get_json(lookup_path)

    gds_url, gds_name, lyp_url, lyp_name = parse_lookup_info(lookup_json)

    dialog.label_Message.text = "REAL: downloading GDS/LYP ..."
    tmpdir = Path(tempfile.mkdtemp(prefix=TEMP_PREFIX))
    gds_path = tmpdir / gds_name
    lyp_path = tmpdir / lyp_name

    client.download(gds_url, gds_path)
    client.download(lyp_url, lyp_path)

    dialog.label_Message.text = f"REAL: opening in KLayout ... ({gds_name})"
    load_design_files(gds=str(gds_path), lyp=str(lyp_path))

    dialog.label_Message.text = f"Done. Opened {gds_name}"
    return

  except Exception as e:
    dialog.label_Message.text    = f"ERROR: {e}"
    dialog.label_Message.enabled = True
    pya.MessageBox.warning("Launcher2", str(e), pya.MessageBox.Ok)
    return

#=======================================================================================
# END SLOT definitions
#=======================================================================================

#=======================================================================================
# BEGIN Helper function / class definitions  ===> see "Launcher.py" by ChatGPT
#=======================================================================================
def load_design_files(gds="", lyp=""):
  global MainWindow

  if not gds == "":
    viewidx = MainWindow.create_view()
    layoutview = MainWindow.view(viewidx)
    layoutview.load_layout(gds)
    if not lyp == "":
      layoutview.load_layer_props(lyp)
    layoutview.zoom_fit()

def eprint(*args):
  print(*args, file=sys.stderr, flush=True)

def parse_xml_for_gds_id(xml_path: Path) -> str:
  tree = ET.parse(str(xml_path))
  root = tree.getroot()
  elem = root.find(".//GdsRefId")
  if elem is not None and elem.text and elem.text.strip():
    return elem.text.strip()
  raise ValueError("GDS reference ID not found in XML")

def cleanup_old_tempdirs(prefix: str = TEMP_PREFIX, days: int = DEFAULT_CLEANUP_DAYS) -> None:
  if os.environ.get("GDSLAUNCHER_NO_CLEANUP", "").strip() in ("1", "true", "TRUE", "yes", "YES"):
    eprint("[launcher] cleanup: disabled by GDSLAUNCHER_NO_CLEANUP")
    return
  base = Path(tempfile.gettempdir())
  now = time.time()
  limit = float(days) * 86400.0
  for p in base.glob(prefix + "*"):
    try:
      if p.is_dir() and (now - p.stat().st_mtime) > limit:
        shutil.rmtree(p, ignore_errors=False)
    except Exception:
      pass

class AuthClient:
  def __init__(self, base_url: str, timeout: int = 30):
    self.base_url = self._normalize_base(base_url)
    self.timeout = timeout
    self._auth_header_value: Optional[str] = None

  @staticmethod
  def _normalize_base(url: str) -> str:
    """Normalize user-entered base URL.

    Accepts:
      - http(s)://host[:port][/optional_base_path]
      - host[:port][/optional_base_path]   (scheme is assumed to be http)

    Also fixes a common mistake where users include a trailing '/api'.
    """
    u = (url or "").strip()
    if not u:
      raise ValueError("Empty base URL")

    p = urlparse(u)
    if not p.scheme:
      u = "http://" + u
      p = urlparse(u)

    # If the user accidentally includes '/api' at the end, strip it.
    path = (p.path or "").rstrip("/")
    if path == "/api":
      p = p._replace(path="")

    u = urlunparse(p)
    if not u.endswith("/"):
      u += "/"
    return u

  def _mk_url(self, path: str) -> str:
    return urljoin(self.base_url, path.lstrip("/"))

  def _request(
    self,
    method: str,
    url: str,
    body: Optional[bytes] = None,
    extra_headers: Optional[dict] = None,
    accept: Optional[str] = None,
  ):
    headers: Dict[str, str] = {}
    if accept:
      headers["Accept"] = accept
    if extra_headers:
      headers.update(extra_headers)
    if self._auth_header_value:
      headers.setdefault("Authorization", self._auth_header_value)
    req = Request(url=url, method=method.upper(), data=body, headers=headers)
    return urlopen(req, timeout=self.timeout)

  @staticmethod
  def _read_http_error(e: HTTPError) -> bytes:
    try:
      return e.read()
    except Exception:
      return b""

  @staticmethod
  def _extract_error_detail(raw: bytes) -> Optional[str]:
    # Try JSON first
    try:
      j = json.loads(raw.decode("utf-8"))
      if isinstance(j, dict):
        for k in ("error", "message", "detail", "reason"):
          v = j.get(k)
          if isinstance(v, str) and v.strip():
            return v.strip()
        # If dict but no known keys, show compact JSON
        s = json.dumps(j, ensure_ascii=False)
        if len(s) > 300:
          s = s[:300] + " ..."
        return s
      # Other JSON types
      s = json.dumps(j, ensure_ascii=False)
      if len(s) > 300:
        s = s[:300] + " ..."
      return s
    except Exception:
      pass

    # Fallback to text
    try:
      s = raw.decode("utf-8", errors="replace").strip()
      if s:
        if len(s) > 300:
          s = s[:300] + " ..."
        return s
    except Exception:
      pass
    return None

  @classmethod
  def _format_http_error(cls, context: str, code: int, raw: bytes) -> str:
    """One-line, user-friendly HTTP error message."""
    detail = cls._extract_error_detail(raw)

    # Auth errors
    if code in (401, 403):
      msg = f"{context}: Authentication failed ({code})"
      if detail:
        msg += f": {detail}"
      return msg

    # Not found
    if code == 404:
      msg = f"{context}: Not found (404)"
      if detail:
        msg += f": {detail}"
      return msg

    # Server-side errors
    if code >= 500:
      # Keep it short; details go to log.
      return f"{context}: Server error ({code}). See log for details."

    # Other client errors
    if detail:
      return f"{context}: HTTP {code}: {detail}"
    return f"{context}: HTTP {code}"

  def login(self, username: str, password: str) -> None:
    login_url = self._mk_url(API_LOGIN)
    body = json.dumps({"username": username, "password": password}).encode("utf-8")
    try:
      resp = self._request(
        "POST",
        login_url,
        body=body,
        extra_headers={"Content-Type": "application/json"},
        accept="application/json",
      )
      raw = resp.read()
    except HTTPError as e:
      raw = self._read_http_error(e)
      raise RuntimeError(self._format_http_error("Login failed", e.code, raw))
    except URLError as e:
      raise RuntimeError(f"Login failed: network error: {e}")

    try:
      j = json.loads(raw.decode("utf-8"))
    except Exception:
      raise RuntimeError(f"Login failed: response is not JSON: {raw[:200]!r}")

    token = None
    for k in ("token", "access_token", "jwt", "id_token"):
      v = j.get(k)
      if isinstance(v, str) and v.strip():
        token = v.strip()
        break
    if not token:
      raise RuntimeError("Login failed: token not found in response")

    self._auth_header_value = f"Bearer {token}"

  def get_json(self, path: str) -> Dict[str, Any]:
    url = self._mk_url(path)
    try:
      with self._request("GET", url, accept="application/json") as resp:
        raw = resp.read()
    except HTTPError as e:
      raw = self._read_http_error(e)
      raise RuntimeError(self._format_http_error(f"GET JSON failed for {url}", e.code, raw))
    except URLError as e:
      raise RuntimeError(f"GET JSON failed: network error for {url}: {e}")

    try:
      return json.loads(raw.decode("utf-8"))
    except Exception:
      raise RuntimeError(f"Invalid JSON response from {url}: {raw[:200]!r}")

  def download(self, url_or_path: str, dest_path: Path) -> None:
    url = url_or_path if url_or_path.startswith(("http://", "https://")) else self._mk_url(url_or_path)
    try:
      with self._request("GET", url) as resp:
        with dest_path.open("wb") as f:
          while True:
            chunk = resp.read(1024 * 256)
            if not chunk:
              break
            f.write(chunk)
    except HTTPError as e:
      raw = self._read_http_error(e)
      raise RuntimeError(self._format_http_error(f"Download failed for {url}", e.code, raw))
    except URLError as e:
      raise RuntimeError(f"Download failed: network error for {url}: {e}")

def parse_lookup_info(lookup_json: Dict[str, Any]) -> Tuple[str, str, str, str]:
  gds_url = lookup_json.get("gdsDownloadUrl")
  gds_name = lookup_json.get("gdsFileName") or "layout.gds"
  lyp_url = lookup_json.get("lypDownloadUrl")
  lyp_name = lookup_json.get("lypFileName") or "layout.lyp"

  if not isinstance(gds_url, str) or not gds_url.strip():
    raise RuntimeError("Lookup response missing gdsDownloadUrl")
  if not isinstance(lyp_url, str) or not lyp_url.strip():
    raise RuntimeError("Lookup response missing lypDownloadUrl")

  return gds_url.strip(), str(gds_name).strip(), lyp_url.strip(), str(lyp_name).strip()

def save_settings(dialog) -> None:
  """
  Save the current settings

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance from which the settings are retrieved.

  Returns
  -------
  None

  settings = pya.QSettings()
  This argument-less QSettings() constructor stores data in:
    👉 the OS-standard user preferences location
    👉 the configuration store used by the KLayout application
  As a result, your "Forum2834_ui" group is saved as part of KLayout’s application settings.

  Conceptually,
    [KLayout]
    └─ Forum2834_ui/
         ├─ xml
         ├─ url
         ├─ username
         ├─ debugmode
         ├─ :
         ├─ :
         └─ geometry

  In macOS,
    MacMini{sekigawa} Launcher1 (1)% defaults read de.klayout.KLayout
    {
      "Forum2834_ui.debugmode" = 0;
      "Forum2834_ui.geometry" = {length = 66, bytes = 0x01d9d0cb 00030000 0000024f 00000187 ... 0000053b 000002c0 };
      "Forum2834_ui.url" = "http://127.0.0.1:8000";
      "Forum2834_ui.username" = simran;
      "Forum2834_ui.xml" = abc;
    }
  """
  settings = pya.QSettings()
  settings.beginGroup("Forum2834_ui")
  try:
    settings.setValue("xml",            dialog.lineEdit_RecipeXML.text.strip())
    settings.setValue("ref_id",         dialog.lineEdit_RefID    .text.strip())
    settings.setValue("debugmode",      dialog.checkBox_Debug    .isChecked())
    settings.setValue("url",            dialog.lineEdit_AppURL   .text.strip())
    settings.setValue("url_stat1",      dialog.label_AppURL      .enabled)
    settings.setValue("url_stat2",      dialog.lineEdit_AppURL   .enabled)
    settings.setValue("username",       dialog.lineEdit_UserName .text.strip())
    settings.setValue("username_stat1", dialog.label_UserName    .enabled)
    settings.setValue("username_stat2", dialog.lineEdit_UserName .enabled)
    settings.setValue("password",       dialog.lineEdit_Password .text.strip())
    settings.setValue("password_stat1", dialog.label_Password    .enabled)
    settings.setValue("password_stat2", dialog.lineEdit_Password .enabled)
    settings.setValue("message",        dialog.label_Message     .text.strip())
    settings.setValue("message_stat1",  dialog.label_Message     .enabled)
    settings.setValue("open_button",    dialog.pushButton_Open   .text.strip())
    settings.setValue("geometry",       dialog                   .saveGeometry())
  finally:
    settings.endGroup()
  settings.sync()

def restore_settings(dialog) -> None:
  """
  Restore the current settings

  Parameters
  ----------
  dialog : pya.QDialog
    The dialog instance to which the settings are restored.

  Returns
  -------
  None
  """
  settings = pya.QSettings()
  settings.beginGroup("Forum2834_ui")
  try:
    xml            = settings.value("xml", "")
    ref_id         = settings.value("ref_id", "")
    debugmode      = settings.value("debugmode", False)
    url            = settings.value("url", "")
    url_stat1      = settings.value("url_stat1", True)
    url_stat2      = settings.value("url_stat2", True)
    user           = settings.value("username", "",)
    user_stat1     = settings.value("username_stat1", True)
    user_stat2     = settings.value("username_stat2", True)
    password       = settings.value("password", "",)
    password_stat1 = settings.value("password_stat1", True)
    password_stat2 = settings.value("password_stat2", True)
    message        = settings.value("message", "")
    message_stat1  = settings.value("message_stat1", True)
    open_button    = settings.value("open_button", "")

    dialog.lineEdit_RecipeXML.text = xml
    if debugmode:
      dialog.checkBox_Debug.checkState = pya.Qt_CheckState.Checked
    else:
      dialog.checkBox_Debug.checkState = pya.Qt_CheckState.Unchecked

    dialog.lineEdit_RefID    .text    = ref_id
    dialog.lineEdit_AppURL   .text    = url
    dialog.label_AppURL      .enabled = url_stat1
    dialog.lineEdit_AppURL   .enabled = url_stat2
    dialog.lineEdit_UserName .text    = user
    dialog.label_UserName    .enabled = user_stat1
    dialog.lineEdit_UserName .enabled = user_stat2
    dialog.lineEdit_Password .text    = password
    dialog.label_Password    .enabled = password_stat1
    dialog.lineEdit_Password .enabled = password_stat2
    dialog.label_Message     .text    = message
    dialog.label_Message     .enabled = message_stat1
    dialog.pushButton_Open   .text    = open_button

    geom = settings.value("geometry")
    if geom is not None:
      dialog.restoreGeometry(geom)
  finally:
    settings.endGroup()

#=======================================================================================
# END Helper function / class definitions
#=======================================================================================

#=======================================================================================
# Bootstrap
#=======================================================================================
def Bootstrap():
  #-------------------------------------------
  # [1] Check KLayout's version
  #-------------------------------------------
  CheckKLayoutVersion()

  #-------------------------------------------
  # [2] Initialize global variables
  #-------------------------------------------
  InitGlobals()

  #-------------------------------------------
  # [3] Register this tool to the main menu
  #-------------------------------------------
  RegisterAction()

#===================================================================================
# Change the start directory
os.chdir(BASE_DIR)

Bootstrap()

#---------------
# End of file
#---------------
