#!/usr/bin/env python3
# -*- coding: utf-8 -*-

from __future__ import annotations

import argparse
import json
import os
from http.server import BaseHTTPRequestHandler, HTTPServer
from pathlib import Path
from urllib.parse import urlparse


BASE_DIR = Path(__file__).resolve().parent
TESTDATA_DIR = BASE_DIR / "testdata"

# Map GDS reference IDs to bundled testdata files.
GDS_MAP = {
    "TEST-GDS-001": {
        "gds": "sample.gds",
        "lyp": "sample.lyp",
    },
    "TEST-GDS-002": {
        "gds": "cobra3b.gds",
        "lyp": "cobra3b.lyp",
    },
}

# Mock credentials (server-side).
MOCK_USER = (os.environ.get("MOCK_USER") or "simran").strip()
MOCK_PASS = (os.environ.get("MOCK_PASS") or "simran").strip()


def _json_bytes(obj) -> bytes:
    return json.dumps(obj, ensure_ascii=False).encode("utf-8")


class Handler(BaseHTTPRequestHandler):
    server_version = "mock_backend/0.3"

    def _send_json(self, code: int, obj):
        data = _json_bytes(obj)
        self.send_response(code)
        self.send_header("Content-Type", "application/json; charset=utf-8")
        self.send_header("Content-Length", str(len(data)))
        self.end_headers()
        self.wfile.write(data)

    def _send_file(self, code: int, path: Path, content_type: str = "application/octet-stream"):
        data = path.read_bytes()
        self.send_response(code)
        self.send_header("Content-Type", content_type)
        self.send_header("Content-Length", str(len(data)))
        self.end_headers()
        self.wfile.write(data)

    def log_message(self, fmt, *args):
        super().log_message(fmt, *args)

    def do_POST(self):
        parsed = urlparse(self.path)
        if parsed.path == "/api/auth/login":
            length = int(self.headers.get("Content-Length", "0"))
            raw = self.rfile.read(length) if length > 0 else b"{}"
            try:
                body = json.loads(raw.decode("utf-8"))
            except Exception:
                self._send_json(400, {"error": "Invalid JSON"})
                return

            user = str(body.get("username", "")).strip()
            pw = str(body.get("password", "")).strip()

            if user == MOCK_USER and pw == MOCK_PASS:
                self._send_json(200, {"token": "mock-token-123"})
            else:
                self._send_json(401, {"error": "Invalid username/password"})
            return

        self._send_json(404, {"error": "Not found"})

    def do_GET(self):
        parsed = urlparse(self.path)

        # Lookup endpoint: returns download URLs for both GDS and LYP.
        if parsed.path.startswith("/api/layouts/by-ref/"):
            gds_ref_id = parsed.path.split("/")[-1].strip()
            if gds_ref_id not in GDS_MAP:
                self._send_json(404, {"error": "Unknown gdsRefId"})
                return

            gds_name = GDS_MAP[gds_ref_id]["gds"]
            lyp_name = GDS_MAP[gds_ref_id]["lyp"]

            self._send_json(
                200,
                {
                    "gdsRefId": gds_ref_id,
                    "gdsFileName": gds_name,
                    "gdsDownloadUrl": f"/download/{gds_ref_id}.gds",
                    "lypFileName": lyp_name,
                    "lypDownloadUrl": f"/download/{gds_ref_id}.lyp",
                },
            )
            return

        # Download endpoint for GDS: /download/<gdsRefId>.gds
        if parsed.path.startswith("/download/") and parsed.path.endswith(".gds"):
            name = parsed.path.split("/")[-1]
            gds_ref_id = name[:-4]  # strip ".gds"
            if gds_ref_id not in GDS_MAP:
                self._send_json(404, {"error": "Unknown gdsRefId"})
                return

            gds_file = TESTDATA_DIR / GDS_MAP[gds_ref_id]["gds"]
            if not gds_file.exists():
                self._send_json(500, {"error": f"Missing server file: {gds_file.name}"})
                return

            self._send_file(200, gds_file, "application/octet-stream")
            return

        # Download endpoint for LYP: /download/<gdsRefId>.lyp
        if parsed.path.startswith("/download/") and parsed.path.endswith(".lyp"):
            name = parsed.path.split("/")[-1]
            gds_ref_id = name[:-4]  # strip ".lyp"
            if gds_ref_id not in GDS_MAP:
                self._send_json(404, {"error": "Unknown gdsRefId"})
                return

            lyp_file = TESTDATA_DIR / GDS_MAP[gds_ref_id]["lyp"]
            if not lyp_file.exists():
                self._send_json(500, {"error": f"Missing server file: {lyp_file.name}"})
                return

            self._send_file(200, lyp_file, "application/octet-stream")
            return

        self._send_json(404, {"error": "Not found"})


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--host", default="127.0.0.1")
    ap.add_argument("--port", type=int, default=8000)
    args = ap.parse_args()

    missing = []
    for rid, ent in GDS_MAP.items():
        for k in ("gds", "lyp"):
            p = TESTDATA_DIR / ent[k]
            if not p.exists():
                missing.append(f"{rid}:{p.name}")
    if missing:
        print("[mock_backend] ERROR: missing testdata files:", ", ".join(missing))
        print(f"[mock_backend] testdata dir = {TESTDATA_DIR}")
        return 2

    httpd = HTTPServer((args.host, args.port), Handler)

    print(f"[mock_backend] serving on http://{args.host}:{args.port}")
    print("[mock_backend] endpoints:")
    print("  POST /api/auth/login")
    print("  GET  /api/layouts/by-ref/<gdsRefId>")
    print("  GET  /download/<gdsRefId>.gds")
    print("  GET  /download/<gdsRefId>.lyp")
    print("[mock_backend] users:")
    print(f"  MOCK_USER={MOCK_USER!r}  MOCK_PASS={MOCK_PASS!r}")
    httpd.serve_forever()


if __name__ == "__main__":
    raise SystemExit(main())
