#!/usr/bin/env python3

#----------------------------------------------------------------------------------------
# Ref. https://www.klayout.de/forum/discussion/2543
# Sub. How to convert gds to dxf (using convert options. Set layerName, layer Color...)
# Posted by: jun
#----------------------------------------------------------------------------------------
# Also,
#   Ref. https://www.klayout.de/forum/discussion/2514
#   Sub. Klayout gds to dxf conversion problem
#   Posted by: wildwolfcj
#----------------------------------------------------------------------------------------
# Author: Kazzz-S
# Dated: 2024-07-09
#----------------------------------------------------------------------------------------
import sys
import os
import optparse
import glob
import klayout.db as db

##
## Parse and get the command line options.
##
## :returns:   The options.
## :rtype:     (dictInOut, [True|False], selected_cell)-tuple
##
def GetOptions():
    usage   = ""
    usage += "---------------------------------------------------------------------------------\n"
    usage += " f2543a.py\n"
    usage += "   To check the number of top cells in given GDS/OASIS file(s)\n"
    usage += "\n"
    usage += "   option & argument : comment on option if any               | default value\n"
    usage += "   -----------------------------------------------------------+----------------\n"
    usage += "   <-i|--input<file>>: GDS/OASIS file(s) to test              | ''\n"
    usage += "       can be a pattern like '*.gds' (Linux); *.gds (Windows) |\n"
    usage += "   [-o|--gendxf]: generate corresponding DXF file(s)          | disabled\n"
    usage += "   [-s|--select<cell>]: take this cell in the 1st input only  | ''\n"
    usage += "   [-?|--?]: print this usage and exit                        | disabled\n"
    usage += "--------------------------------------------------------------+------------------\n"

    p = optparse.OptionParser( usage=usage )
    p.add_option( '-i', '--input',
                  dest='input_files',
                  help='GDS/OASIS file(s) to test' )

    p.add_option( '-o', '--gendxf',
                  action='store_true',
                  dest='gendxf',
                  help='generate corresponding DXF file(s)' )

    p.add_option( '-s', '--select',
                  dest='selected_cell',
                  help='select a single cell in the 1st input' )

    p.add_option( '-?', '--?',
                  action='store_true',
                  dest='checkusage',
                  default=False,
                  help='print this usage and exit' )

    p.set_defaults( input_files   = "",
                    gendxf        = False,
                    selected_cell = "" )

    opt, args = p.parse_args()
    if opt.checkusage:
        print(usage)
        quit()

    dictInOut = dict()
    for file in glob.glob(opt.input_files):
        f, e = os.path.splitext(os.path.basename(file))
        if e in [ ".gds", ".gds2", ".oas" ]:
            dictInOut[file] = "%s.dxf" % f

    if len(dictInOut) == 0:
        print( "! No GDS/OASIS file is passed" )
        print(usage)
        sys.exit(0)
    else:
        return (dictInOut, opt.gendxf, opt.selected_cell)

##
## Check the number of top cell(s), then generate DXF file(s) on demand.
##
## :param      dictInOut:  The dictionary of input- and output-files
## :type       dictInOut:  key=input GDS/OASIS file name; value=output DXF file name
## :param      gendxf:     The gendxf flag
## :type       gendxf:     True to generate DXF; False, otherwise
## :param      tcName:     top cell name in the 1st input
## :type       tcName:     string
##
def CheckTopCells( dictInOut, gendxf, tcName ):
    for design in sorted(dictInOut.keys()):
        dxf = dictInOut[design]
        layout = db.Layout()
        layout.read(design)
        dbuIn = layout.dbu # most likely 0.001[um]= 1.0[nm]
        top_cells = layout.top_cells()
        top_cell_names = [ cell.name for cell in top_cells]

        if len(top_cell_names) > 1:
            headings = [ "Input GDS <%s> has (%d) top cells:" % (design, len(top_cell_names)),
                         "  !!! Consider unifying the top cells to eliminate ambiguity!!!"
                       ]
        else:
            headings = [ "Input GDS <%s> has a single (%d) top cell:" % (design, len(top_cell_names)),
                         ""
                       ]

        print(headings[0])
        for name in top_cell_names:
            print( "    --> %s" % name )
        print(headings[1])

        if gendxf:
            # The -s|--select option was used for the 1st input file.
            if tcName != "":
                if not tcName in top_cell_names:
                    print( "  ??? Specified top cell <%s> not found in <%s>." % (tcName, design) )
                    sys.exit(1)
                else:
                    so = db.SaveLayoutOptions() # SaveLayoutOptions object
                    so.format = "DXF"           #   format is 'DXF'
                    so.dbu = 0.001              #   dbu [um]
                    so.scale_factor = 0.001     #   assume CAD tool's drawing unit is [mm]
                    so.dxf_polygon_mode = 0     #   0 (write POLYLINE entities),
                                                #   1 (write LWPOLYLINE entities),
                                                #   2 (decompose into SOLID entities),
                                                #   3 (write HATCH entities), or
                                                #   4 (write LINE entities)
                    so.select_cell( layout.cell(tcName).cell_index() )
                    layout.write( dxf, so )
                    print( "  >>> With top cell [%s], generated <%s>" % (tcName, dxf) )
                    print( "" )
                    sys.exit(0)
            else:
                # Single top cell
                if len(top_cell_names) == 1:
                    so = db.SaveLayoutOptions()
                    so.format = "DXF"
                    so.dbu = 0.001
                    so.scale_factor = 0.001
                    so.dxf_polygon_mode = 0
                    layout.write( dxf, so )
                    print( "  >>> With top cell [%s], generated <%s>" % (top_cell_names[0], dxf) )
                    print( "" )
                # Multiple top cells
                else:
                    for tc in top_cells:
                        so = db.SaveLayoutOptions()
                        so.format = "DXF"
                        so.dbu = 0.001
                        so.scale_factor = 0.001
                        so.dxf_polygon_mode = 0
                        so.select_cell( tc.cell_index() )
                        tcdxf = "__%s__%s" % (tc.name, dxf)
                        layout.write( tcdxf, so )
                        print( "  >>> With top cell [%s], generated <%s>" % (tc.name, tcdxf) )
                    print( "" )


##
## The main function
##
def Main():
    dictInOut, gendxf, tcName = GetOptions()
    CheckTopCells( dictInOut, gendxf, tcName )


if __name__ == "__main__":
    Main()
